const { app } = require('electron');
const fs = require('fs');
const path = require('path');
const crypto = require('crypto');
const logger = require('./logger');

const settingsPath = path.join(app.getPath('userData'), 'settings.json');
const DEFAULT_ANALYTICS = {
  optedIn: true,
  userId: null,
  lastConsentUpdatedAt: null,
  sessionRecording: {
    enabled: false,
    sampleRate: 0.1
  }
};
const DEFAULT_SETTINGS = {
  monitoredDirectories: [],
  analytics: DEFAULT_ANALYTICS
};

function clone(value) {
  return JSON.parse(JSON.stringify(value));
}

function ensureDirectoryObjects(monitoredDirectories) {
  if (!Array.isArray(monitoredDirectories)) {
    return { value: [], changed: true };
  }

  if (monitoredDirectories.length === 0) {
    return { value: monitoredDirectories, changed: false };
  }

  if (typeof monitoredDirectories[0] === 'string') {
    const converted = monitoredDirectories.map((dirPath) => ({
      path: dirPath,
      name: path.basename(dirPath)
    }));
    return { value: converted, changed: true };
  }

  return { value: monitoredDirectories, changed: false };
}

function ensureAnalyticsSettings(analytics) {
  const normalized = { ...DEFAULT_ANALYTICS, ...(analytics || {}) };
  let changed = false;

  if (normalized.optedIn !== true) {
    normalized.optedIn = true;
    changed = true;
  }

  if (!normalized.userId) {
    normalized.userId = crypto.randomUUID();
    changed = true;
  }

  if (!normalized.sessionRecording || typeof normalized.sessionRecording !== 'object') {
    normalized.sessionRecording = { ...DEFAULT_ANALYTICS.sessionRecording };
    changed = true;
  } else {
    if (typeof normalized.sessionRecording.enabled !== 'boolean') {
      normalized.sessionRecording.enabled = Boolean(normalized.sessionRecording.enabled);
      changed = true;
    }

    let sampleRate = Number(normalized.sessionRecording.sampleRate);
    if (!Number.isFinite(sampleRate) || sampleRate < 0 || sampleRate > 1) {
      sampleRate = DEFAULT_ANALYTICS.sessionRecording.sampleRate;
      changed = true;
    } else {
      sampleRate = Math.max(0, Math.min(1, sampleRate));
      if (sampleRate !== normalized.sessionRecording.sampleRate) {
        changed = true;
      }
    }
    normalized.sessionRecording.sampleRate = sampleRate;
  }

  return { value: normalized, changed };
}

function normalizeSettings(rawSettings) {
  const base = typeof rawSettings === 'object' && rawSettings !== null ? clone(rawSettings) : {};
  let changed = false;

  const directoriesResult = ensureDirectoryObjects(base.monitoredDirectories);
  const analyticsResult = ensureAnalyticsSettings(base.analytics);

  const normalized = {
    monitoredDirectories: directoriesResult.value,
    analytics: analyticsResult.value
  };

  if (directoriesResult.changed || analyticsResult.changed) {
    changed = true;
  }

  return { normalized, changed };
}

function writeSettingsFile(settings) {
  fs.writeFileSync(settingsPath, JSON.stringify(settings, null, 2));
}

function loadSettings() {
  let rawSettings = {};
  let fromDisk = false;

  try {
    if (fs.existsSync(settingsPath)) {
      const fileContents = fs.readFileSync(settingsPath, 'utf8');
      rawSettings = JSON.parse(fileContents);
      fromDisk = true;
    }
  } catch (error) {
    logger.error('Error loading settings:', error);
  }

  const { normalized, changed } = normalizeSettings(rawSettings);

  if (changed || !fromDisk) {
    try {
      writeSettingsFile(normalized);
    } catch (error) {
      logger.error('Error saving settings:', error);
    }
  }

  return normalized;
}

function saveSettings(settings) {
  try {
    const { normalized } = normalizeSettings(settings);
    writeSettingsFile(normalized);
    return normalized;
  } catch (error) {
    logger.error('Error saving settings:', error);
    throw error;
  }
}

function getAnalyticsSettings() {
  const settings = loadSettings();
  return clone(settings.analytics);
}

function updateAnalyticsPreferences({
  optedIn,
  rotateUserId,
  sessionRecordingEnabled,
  sessionRecordingSampleRate
} = {}) {
  const settings = loadSettings();
  let changed = false;

  if (settings.analytics.optedIn !== true) {
    settings.analytics.optedIn = true;
    settings.analytics.lastConsentUpdatedAt = new Date().toISOString();
    changed = true;
  } else if (optedIn === true) {
    settings.analytics.lastConsentUpdatedAt = new Date().toISOString();
    changed = true;
  }

  if (rotateUserId === true) {
    settings.analytics.userId = crypto.randomUUID();
    settings.analytics.lastConsentUpdatedAt = new Date().toISOString();
    changed = true;
  }

  if (typeof sessionRecordingEnabled === 'boolean'
    && settings.analytics.sessionRecording
    && settings.analytics.sessionRecording.enabled !== sessionRecordingEnabled) {
    settings.analytics.sessionRecording.enabled = sessionRecordingEnabled;
    changed = true;
  }

  if (sessionRecordingSampleRate != null && settings.analytics.sessionRecording) {
    const numeric = Number(sessionRecordingSampleRate);
    if (Number.isFinite(numeric)) {
      const clamped = Math.max(0, Math.min(1, numeric));
      if (clamped !== settings.analytics.sessionRecording.sampleRate) {
        settings.analytics.sessionRecording.sampleRate = clamped;
        changed = true;
      }
    }
  }

  if (changed) {
    saveSettings(settings);
  }

  return clone(settings.analytics);
}

function setAnalyticsOptIn() {
  return updateAnalyticsPreferences({ optedIn: true });
}

function rotateAnalyticsUserId() {
  return updateAnalyticsPreferences({ rotateUserId: true });
}

module.exports = {
  settingsPath,
  loadSettings,
  saveSettings,
  getAnalyticsSettings,
  updateAnalyticsPreferences,
  setAnalyticsOptIn,
  rotateAnalyticsUserId
};

